# Author: Núria Serra Serra
# Date: April 2017

############################################################################################################

# CALCULATION OF THE PAIRWISE GEOGRAPHIC DISTANCES BETWEEN SAMPLING SITES FOR K AND G POPULATIONS SEPARATELY
 
# Using marmap(which uses data from NOAA to calculate minimal least cost distances between 2 locations) and fossil (Eucledian distances)

############################################################################################################



remove(list=ls())

library(marmap) 

###########
### Open file with coordinates, replace commas by dots and make coordinate files for only K and G pops:
###########

df_coord = read.csv("coordinates_samplingsites_lonlat.csv", sep=";")

df_coord$Lon = as.numeric(gsub(",", ".", gsub("\\.", "", df_coord$Lon)))
df_coord$Lat = as.numeric(gsub(",", ".", gsub("\\.", "", df_coord$Lat)))

names(df_coord)[1] = "population"
df_coord$population

coord_G = df_coord[df_coord$population %in% c("G-AF", "G-AS", 
                                              "G-BB", "G-BV", "G-GB", 
                                              "G-NB", "G-RX", "G-SG", "G-SK", 
                                              "Snackebackebukten", 
                                              "Torgestad"),]

coord_K = df_coord[df_coord$population %in% c("K-BK", "K-K", "K-KR", "K-LD", 
                                              "K-NG", "K-NI", "K-OK", "K-ON", 
                                              "K-RT", "K-SK", "K-SO"),]


# Drop levels:
coord_K$population = droplevels(coord_K$population)
coord_G$population = droplevels(coord_G$population)

levels(coord_G$population)[10] = "G-SN"
levels(coord_G$population)[11] = "G-T"

coord_G$population = droplevels(coord_G$population)

###########
### Get the data from NOAA for all sites and for K and G separately.
###########

Formas <- getNOAA.bathy(lon1 = 9, lon2 = 13,
                        lat1 = 55, lat2 = 60, resolution = 1)

G_map <- getNOAA.bathy(lon1 = 11, lon2 = 12,
                        lat1 = 58, lat2 = 58.7, resolution = 1)

K_map <- getNOAA.bathy(lon1 = 11.3, lon2 = 12,
                        lat1 = 57.5, lat2 = 58.3, resolution = 1)


###########
### Plotting the sampling points in the map for all/K/G sampling sites:
###########

blues <- c("lightsteelblue4", "lightsteelblue3",
           "lightsteelblue2", "lightsteelblue1")
greys <- c(grey(0.6), grey(0.93), grey(0.99))

## All
plot(Formas, image = TRUE, land = TRUE, lwd = 0.03,
     bpal = list(c(0, max(Formas), greys),
                 c(min(Formas), 0, blues)))
points(df_coord$Lon, df_coord$Lat, pch=16, col="green")

## G
png(filename = "G_withland_moved.png", 
    width=12, height=12, units="cm", pointsize=12, bg="white", res=600)

plot(G_map, image = TRUE, land = TRUE, lwd = 0.03,
     bpal = list(c(0, max(G_map), greys),
                 c(min(G_map), 0, blues)))
points(coord_G$Lon, coord_G$Lat, pch=16, col="green")

## K
plot(K_map, image = TRUE, land = TRUE, lwd = 0.03,
     bpal = list(c(0, max(K_map), greys),
                 c(min(K_map), 0, blues)))
points(coord_K$Lon, coord_K$Lat, pch=16, col="red")


###########
### Calculating the least path between sampling sites (for K and G separately)
###########

# Firstly, compute a transition matrix to be used by lc.dist() to compute least cost distances between locations.
# The transition object generated by trans.mat contains the probability of transition from one cell of a bathymetric grid
# to adjacent cells, and depends on user defined parameters.
# Default values of trans.mat for min and max depths ensure that the path computed will be the shortest possible at sea 
# avoiding land masses.


# PROBLEM: some of the coordinates for the sampling sites of the G fjord are above 0, which makes the
# distances from them to the rest extremely large (milions of km).
# If I only expand the range of probable dispersal (min.depth=100) then the distances I obtain are almost 
# the same as Eucledian distances, because they allow Zostera to disperse above ground.
#SOLUTION: move these sampling points to a place nearby where the depth is 0 or below.


#Create df to import in lc.dist:
G_sites = data.frame(coord_G$Lon, coord_G$Lat)
names(G_sites) = c("x", "y")

K_sites = data.frame(coord_K$Lon, coord_K$Lat)
names(K_sites) = c("x", "y")

all_sites = data.frame(df_coord$Lon, df_coord$Lat)
names(all_sites) = c("x", "y")


#Calculate transition matrices
trans_G = trans.mat(G_map)
trans_K = trans.mat(K_map)
trans_all = trans.mat(Formas)

trans_G_above100 = trans.mat(G_map, min.depth = -100) #From deepest until 100m below 0
trans_G_below20 = trans.mat(G_map, min.depth = 20) #From deepest until 20m above sea level
trans_G_below7 = trans.mat(G_map, min.depth = 7) #From deepest until 7m above sea level


plot(raster(trans_G), main="No depth constraint")
plot(raster(trans_K), main="No depth constraint")
plot(raster(trans_G_above100), main="Constraint in waters shallower than 100 - only below 100m")
plot(raster(trans_G_below7), main="From the deepest until 7m above water")
plot(raster(trans_G_below20), main="From the deepest until 20m above water")

# For G, change some sampling locations to places that are at 0m or below ground.
get.depth(G_map)

#G-SK: from 98m (11.69950,58.43083) to 4m (11.68121, 58.44556)
coord_G[1,]
coord_G[1,2] = 11.68121
coord_G[1,3] = 58.44556

G_sites[1,1] = 11.68121
G_sites[1,2] = 58.44556

#Snack: from 17m (11.56000 58.36000) to  1m (11.56892 58.35308)
coord_G[10,]
coord_G[10,2] = 11.56892
coord_G[10,3] = 58.35308

G_sites[10,1] = 11.56892
G_sites[10,2] = 58.35308

#G-AF: from 19m (11.41122 58.41630) to 3m (11.39441 58.41557)
coord_G[5,]
coord_G[5,2] = 11.39441
coord_G[5,3] = 58.41557

G_sites[5,1] = 11.39441
G_sites[5,2] = 58.41557


#G-BV: from 18m (11.37793, 58.36555) to 5m (11.42616 58.38089)
coord_G[8,]
coord_G[8,2] = 11.42616
coord_G[8,3] = 58.38089

G_sites[8,1] = 11.42616
G_sites[8,2] = 58.38089


#G-RX: from 12m (11.44440 58.36442) to 5m (11.42993 58.36771)
coord_G[9,]
coord_G[9,2] = 11.42993
coord_G[9,3] = 58.36771

G_sites[9,1] = 11.42993
G_sites[9,2] = 58.36771

#G-AS: from 7m (11.37793, 58.36555) to 
coord_G[6,]


#Output of lc.dist as a list of paths (2-col matrices of routes) between paits of locations
lc_path_G = lc.dist(trans_G, G_sites, res = "path")
lc_path_G_above100 = lc.dist(trans_G_above100, G_sites, res = "path")
lc_path_G_below20 = lc.dist(trans_G_below20, G_sites, res = "path")
lc_path_G_below7 = lc.dist(trans_G_below7, G_sites, res = "path")

lc_path_K = lc.dist(trans_K, K_sites, res = "path")
lc_path_all = lc.dist(trans_all, all_sites, res="path")


#Plotting the paths:

# G:
png(filename = "G_map_withleastcost.png", 
    width=12, height=12, units="cm", pointsize=12, bg="white", res=600)


plot(G_map, xlim = c(11,12), ylim = c(58, 58.7), land=TRUE,
     deep = c(-500, -20, 0, 01, 20), shallow = c(-20, 0, 0, 20, 200),
     col=c("dodgerblue", "dodgerblue3", "black", "gray85", "gray60"), step = c(20,5,1,5, 25),
     lty = c(1, 1, 1, 1, 1), lwd = c(1.2, 0.8, 2,  1, 1.2),
     draw = c(TRUE, TRUE, TRUE, TRUE, TRUE))

points(G_sites, pch = 21, col = "green", bg = col2alpha("green", .9),
       cex = 1)
text(G_sites[,1], G_sites[,2], lab = coord_G$population,
     pos = c(3, 4, 1, 2), col = "black", cex=0.6)
lapply(lc_path_G, lines, col = "orange", lwd = 2, lty = 1) -> dummy
# lapply(lc_path_G_below7, lines, col = "blue", lwd = 1, lty = 1) -> dummy
# lapply(lc_path_G_below20, lines, col = "red", lwd = 1, lty = 1) -> dummy
# lapply(lc_path_G_above100, lines, col = "black", lwd = 1, lty = 1) -> dummy

dev.off()
dev.list()
dev.set(which = 2)
dev.cur()

#K
png(filename = "K_map_withleastcost.png", 
    width=12, height=12, units="cm", pointsize=12, bg="white", res=600)

plot(K_map, xlim = c(11.3,12), ylim = c(57.5, 58.3), land=TRUE,
     deep = c(-500, -20, 0, 01, 20), shallow = c(-20, 0, 0, 20, 200),
     col=c("dodgerblue", "dodgerblue3", "black", "gray85", "gray60"), step = c(20,5,1,5, 25),
     lty = c(1, 1, 1, 1, 1), lwd = c(1.2, 0.8, 2,  1, 1.2),
     draw = c(TRUE, TRUE, TRUE, TRUE, TRUE))
points(K_sites, pch = 21, col = "red", bg = col2alpha("red", .9),
       cex = 1)
text(K_sites[,1], K_sites[,2], lab = coord_K$population,
     pos = c(3, 4, 1, 2), col = "black", cex=0.8)
lapply(lc_path_K, lines, col = "orange", lwd = 2, lty = 1) -> dummy
dev.off()
dev.list()
dev.set(which = 2)
dev.cur()
#All
png(filename = "All_map_withleastcost.png", 
    width=12, height=12, units="cm", pointsize=12, bg="white", res=600)

plot(Formas, xlim = c(9,13), ylim = c(55, 60), land=TRUE,
     dee
p = c(-500, -20, 0, 01, 20), shallow = c(-20, 0, 0, 20, 200),
     col=c("dodgerblue", "dodgerblue3", "black", "gray85", "gray60"), step = c(20,5,1,5, 25),
     lty = c(1, 1, 1, 1, 1), lwd = c(1.2, 0.8, 2,  1, 1.2),
     draw = c(TRUE, TRUE, TRUE, TRUE, TRUE))
points(all_sites, pch = 21, col = "red", bg = col2alpha("red", .9),
       cex = 1)
text(all_sites[,1], all_sites[,2], lab = coord_K$population,
     pos = c(3, 4, 1, 2), col = "black", cex=0.8)
lapply(lc_path_all, lines, col = "orange", lwd = 2, lty = 1) -> dummy
dev.off()
dev.list()
dev.set(which = 2)
dev.cur()

# Compute a KILOMETRIC distance matrix with pairwise genetic distances between sampling sites 
# (lc.dist with "dist") in lc.dist and with earth.dist of fossil package
library(fossil)
dist0_G = round(earth.dist(G_sites), 0)
dist1_G = lc.dist(trans_G, G_sites, res = "dist")
dist2_G = lc.dist(trans_G_above100, G_sites, res = "dist")
dist3_G = lc.dist(trans_G_below20, G_sites, res = "dist")
dist4_G = lc.dist(trans_G_below7, G_sites, res = "dist")

dist0_K = round(earth.dist(K_sites), 0)
dist1_K = lc.dist(trans_K, K_sites, res = "dist")

# dist0_all = round(earth.dist(all_sites), 0)
# dist1_all = lc.dist(trans_all, all_sites, res = "dist")

library(MASS)
write.matrix(dist0_G, "G_earthdist_mat.txt")
write.matrix(dist1_G, "G_lcdist_mat.txt")
write.matrix(dist3_G, "G_lcdist_mat_below20.txt")
write.matrix(dist4_G, "G_lcdist_mat_below7.txt")

write.matrix(dist0_K, "K_earthdist_mat.txt")
write.matrix(dist1_K, "K_lcdist_mat.txt")
